#############################################################################
#############################################################################
#
# latexEngine.tcl (called from latex.tcl)
#
# The LaTeX engine
#
#############################################################################
#
# Old Author:  Tom Scavo <trscavo@syr.edu>
# New Author:  Vince Darley <vince@santafe.edu>
#
#############################################################################
#############################################################################
proc latexEngine.tcl {} {}


#--------------------------------------------------------------------------
# Mark Menu:
#--------------------------------------------------------------------------

# Bugs:  -- won't add a section (in a different chapter) with the same name.
#        -- should ignore comments, but doesn't.
#        -- a sectioning command must be on a line by itself.
proc TeX::MarkFile {} {
  global markCommandsNotStructure
  if {[lsearch -exact $markCommandsNotStructure [file extension [win::CurrentTail]]] != -1} {
    return [TeX::Sty_MarkFile]
  }
  set pos [minPos]
  set leader {}
  # Vince's improvement (but doesn't allow embedded braces):
  # JEG improvement (add's includegraphics)
  set exp {\\((sub)*section|chapter|input|include|usepackage|includegraphics)(\[[^]]*\]|\*)?\{}
  set sectioning ""
  while {![catch {search -s -f 1 -r 1 -m 0 -i 0 $exp $pos} res] } {
    set start [lindex $res 0]
    set braceStart [lindex $res 1]
    set end [matchIt "\{" $braceStart]
    set braceContents [getText $braceStart $end]
    set cmd [getText $start $braceStart]
    
    regsub -all "\\\\label\{\[^{}\]\}" $braceContents "" braceContents
    regsub -all "\[{}\]" [string trim $braceContents] "" braceContents
    
    if {[string length $braceContents]} {
      set item ""
      set cmdName [extractCommandName $cmd]
      if { [regexp {((sub)*)section} $cmdName dummy prefix] } {
        # indent \(sub)*section or \(sub)*section*:
        set spaces [expr 2 * [string length $prefix] / 3]
        append item $leader [format "%$spaces\s" ""]
        set sectioning  [format "%$spaces\s" ""]
      } elseif { [regexp {chapter} $cmdName] } {
        # if \chapter or \chapter*, indent next \(sub)*section:
        set leader {  }
        set sectioning ""
      } elseif { [regexp {includegraphics} $cmdName] } then {
        # mark \includegraphics (JEG):
        append item "${leader}${sectioning}  "
      } else {
        # mark \input, \include, and \usepackage (VD):
        append item ""
      }
      # remove all superfluous whitespace (WTP):
      regsub -all "\[\ \r\n\t\]\+" $braceContents { } braceContents
      # limit the width of the menu item (WTP):
      if {([info tclversion] < 8.0) && ([string length $braceContents] > 30)} { 
        set braceContents "[string range $braceContents 0 26]" 
      }
      # build the menu item:
      append item $braceContents
      # create the mark:
      while {[lcontains marks $item]} {
        append item " "
      }
      lappend marks $item
      setNamedMark $item [lineStart [pos::math $start - 1]] $start $start
    }
    set pos [pos::math $end + 1]
  }
}

proc TeX::Sty_MarkFile {} {
  set pos [minPos]
  set leader {}
  set exp {\\((re)?newcommand(\*)?\{|(e|g|x)?def|DeclareRobustCommand(\*)?\{|providecommand(\*)?\{)\\([^[\#\\\{\}\r\n%]*)}
  while { ![catch {search -s -f 1 -r 1 -m 0 -i 0 $exp $pos} res] } {
    set start [lindex $res 0]
    set end [lindex $res 1]
    set cmd [getText $start $end]
    if { [regexp -- $exp $cmd d d d d d d d item] } {
      # limit the width of the menu item (WTP):
      if { [string length $item] > 30 } { set item "[string range $item 0 26]" }
      # save the mark:
      lappend items [list $item $start]
    }
    set pos [pos::math $end + 1]
  }
  foreach i [lsort -ignore $items] {
    set start [lindex $i 1]
    setNamedMark [lindex $i 0] [lineStart [pos::math $start - 1]] $start $start
  }
}

proc TeX::parseFuncs {} {
  global funcExpr parseExpr
  
  set m {}
  set pos [minPos]
  while {[set res [search -s -f 1 -r 1 -i 0 -n $funcExpr $pos]] != ""} {
    set text [getText [lindex $res 0] [pos::math [nextLineStart [lindex $res 1]] - 1]]
    if {[regexp -- $parseExpr $text dummy word]} {
      set num [regsub -all sub $text sub dummy]
      lappend m "[format %${num}s {}]$word" [lindex $res 0]
    }
    set pos [lindex $res 1]
  }
  return $m
}

#--------------------------------------------------------------------------
# Command-double-clicking:
#--------------------------------------------------------------------------
# In TeX mode, use cmd-double-clicks to follow references and citations,
# or open input files.
#
# (originally written by Tom Pollard and Andreas Amann)
#
# Remaining bugs:
#   - search is successful even if the pattern is commented out
#     (which is a bug or a feature, depending how you look at it)

# Extended to work with filesets and with better bib code
proc TeX::DblClick {from to shift option control} {
  if {[file extension [win::Current]] == ".blg" \
   || [win::StripCount [win::Current]] == "* bibtex log *"} {
    Bib::blgDoubleClick $from $to
    return
  }
  global TeXmodeVars
  
  # CONTROL -- send to url for LaTeX command help
  if {$control != 0} {
    select $from $to
    regsub {\\} [getSelect] {} cmd
    url::execute $TeXmodeVars(latexHelp)$cmd
    message "\"$cmd\" sent to $TeXmodeVars(latexHelp)"
    return
  }
  
  # Extend selection to largest string delimited by commas or curly-braces
  set text [string trim [eval getText [TeXExtendArg $from $to]]]
  
  # Set $cmd to TeX command for which the selection is an argument, but
  # only if user clicked on a valid command argument.
  set cmd {}
  if {[set mtch [findCommandWithParts $from 0]] != ""} {
    set beg [lindex $mtch 0]
    set arg [lindex $mtch 2]
    set end [lindex $mtch 3]
    # Make sure the user clicked within a TeX argument and not
    # on the command name itself
    if {[pos::compare $from > $arg] && [pos::compare $to < $end]} {
      set cmd [extractCommandName [getText $beg $arg]]
    }
  }
  
  switch -- $cmd {
    "@input" -
    "input" - 
    "InputIfFileExists" {openTeXFile $text ".tex"}
    "include" {openTeXFile "${text}.tex"}
    "documentclass" -
    "LoadClass" {openTeXFile "${text}.cls"}
    "usepackage" -
    "RequirePackage" {openTeXFile "${text}.sty"}
    "bibliography" {Bib::openFile "${text}.bib"}
    "bibliographystyle" {Bib::openFile "${text}.bst"}
    default {
      # \cite, \nocite, etc.
      if {[lsearch -exact $TeXmodeVars(citeCommands) $cmd] >= 0} {
        TeX_dblClickCitation $text
        return
        
        # \ref, \pageref, etc.
      } elseif {[lsearch -exact $TeXmodeVars(refCommands) $cmd] >= 0} {
        set labelPat {\\label\{}
        # Check for \label in current window/fileset
        if {[TeX_selectPatternInFileOrSet "$labelPat$text\}" "tex"]} {
          return
        }
        # Only gets here if the search failed
        beep
        returnToBookmark
        message {No matching \label found}
        # box-making macro ($boxMacroNames)
      } elseif { [lsearch -exact $TeXmodeVars(boxMacroNames) $cmd] > -1 } { 
        if {[file extension $text] == ""} {
          foreach ext {.eps .ps .epsf} {
            if {[set f [findTeXFile $text$ext]] != ""} {
              break
            }
          }
        } else {
          set f [findTeXFile $text]
        }
        if { $f != "" } {
          switch -- [string tolower [file extension $f]] {
            ".ps" -
            ".eps" -
            ".epsf" {
              switch [buttonAlert \
               "Do you wish to view or edit \"[file tail $f]\"?" \
               "View" "Edit" "Cancel" \
               ] {
                "View" {viewPSFile $f}
                "Edit" {
                  file::openQuietly $f
                  message $f
                }
              }
            }
            default {
              file::openQuietly $f
              message $f
            }
          }
        }
        # Other
      } else {
        select $from $to
        message {Command-double-click on the required argument of an underlined LaTeX command}
      }
    }
  }
}

## 
 # -------------------------------------------------------------------------
 # 
 # "TeX_selectPatternInFileOrSet" --
 # 
 #  Useful procedure which will search for a given pattern in all files in
 #  the current fileset (or the fileset of the given window) if such a 
 #  fileset exists, and otherwise it will just look at the current tex
 #  window.  The 'types' argument may be '*', 'bib' or 'tex' depending
 #  upon which types of file you wish to search in the fileset.
 #  
 #  Returns 1 if successful, else 0.
 # -------------------------------------------------------------------------
 ##
proc TeX_selectPatternInFileOrSet {pat {types "*"} {win ""}} {
  if {$win == ""} {
    set win [win::Current]
  }
  if {[set fset [isWindowInFileset $win "tex"]] == ""} {
    if {[selectPatternInFile $win "$pat"]} {
      return 1
    }
  } else {
    foreach filename [fileset::tex::listFiles $fset $types] {
      if {[selectPatternInFile $filename "$pat"]} {
        return 1
      }			
    }
  } 
  return 0
}

proc TeX_dblClickCitation {text} {
  global AllTeXSearchPaths
  # Quote regexp-active characters in case we use $text in a regexp search
  set qtext [quote::Regfind $text]
  set bibPat "\\bibliography\{"
  set bibTopPat {@([a-zA-Z]+)[\{\(][ 	]*}
  
  # Check first in the current window, either for a bib entry or a \bibitem
  if {[TeX_findBibItem $qtext [list [win::Current]]]} {
    return
  }
  # If we're in a fileset...
  if {[set theFileset [isWindowInFileset "" "tex"]] != ""} {
    # search all the files in that fileset for bib entries or \bibitems
    if {[TeX_findBibItem $qtext [fileset::tex::listFiles $theFileset "bib"]]} { 
      return
    }
    if {[TeX_findBibItem $qtext [fileset::tex::listFiles $theFileset "tex"]]} { 
      return
    }
    beep
    message "can't find \"$text\" in the .bib file(s) or in \\thebibliography"
  } else {
    # if no fileset, and we didn't find it above, look for all the
    # included bibliographies, and search them:
    if {![catch {search -f 0 -r 0 -s -m 0 $bibPat [maxPos]} mtch]} {
      # Get ALL the bib file names:
      set beg [lindex $mtch 1]
      set end [matchIt "\{" $beg]
      set bibnames "[split [getText $beg $end] ,]"
      # Check every file:
      foreach fname $bibnames {
        if {[selectPatternInFile [absolutePath "${fname}.bib"] $bibTopPat$qtext]} {
          return
        }
      }
      # Build the TeX search path:
      TeXEnsureSearchPathSet
      # Check every file in the TeX search path:
      foreach fname $bibnames {
        foreach folder $AllTeXSearchPaths {
          if {[selectPatternInFile [file join $folder ${fname}.bib] $bibTopPat$qtext]} {
            return
          }
        }
      }
      beep
      message "can't find \"$text\" in the .bib file(s)"	
    } else {
      beep
      message "can't find a \\bibliography"
    }
  }
  # Search through all the bibliographies using the bibIndex
  # If we find the entry,
  # prompt to insert a new database in the list of bibliographies.
  set c [TeX_currentBaseFile [win::Current]]
  if {![catch {Bib::gotoEntryFromIndex $text}]} {
    if {[askyesno "This entry is in a bibliography file which isn't referenced.  Shall I add it to this document's list of bibliographies?"] == "yes"} {
      Bib::insertNewBibliography $c [file::baseName [win::Current]]
    }
    return
  }
  # There's no matching entry.  Give a list of possibilities.
  Bib::noEntryExists $text [TeX_currentBaseFile]
}

proc TeX_findBibItem {qtext fnames} {
  set theBibPat "\\\\begin\{thebibliography"
  set bibitemPat {\\bibitem(\[[^\]*\])?\{}
  set bibTopPat {@([a-zA-Z]+)[\{\(][ 	]*}
  foreach filename $fnames {
    if ![file exists $filename] { continue }
    switch -- [file extension $filename] {
      ".tex" -
      ".ltx" {
        # Check first for a \thebibliography environment...
        set searchResult [searchInFile $filename $theBibPat 1]
        if {[pos::compare [set pos [lindex $searchResult 0]] >= [minPos]]} {
          # ...then for the \bibitem, which must be after
          # \thebibliography declaration to make any sense
          if {[selectPatternInFile $filename "$bibitemPat$qtext\}" $pos]} {
            return 1
          }
        }
      }
      ".bib" {
        if {[selectPatternInFile $filename $bibTopPat$qtext]} {
          return 1
        }
      }
    }
  }
  return 0
}

## 
 # -------------------------------------------------------------------------
 # 
 # "TeX::RequirePackage" --
 # 
 #  Return 1 unless the package wasn't included and the user hit 'cancel',
 #  in which case we return 0.
 # -------------------------------------------------------------------------
 ##
proc TeX::RequirePackage {pkg} {
  # search the document/fileset to ensure we have a 'usepackage'
  # or 'RequirePackage' command for this package.
  
  set filename [TeX_currentBaseFile]
  set pat {\\(usepackage|RequirePackage)}
  append pat "\{$pkg\}"
  if {[file::searchFor $filename $pat 0] == ""} {
    # search through sub-packages
    set pat2 {^[^\r\n]*\\(usepackage|RequirePackage)\{([^\}]+)\}([^\r\n]*)$}
    set subpkgs [file::findAllInstances $filename $pat2 1]
    foreach f $subpkgs {
      if ![catch {set sty [findTeXFile $f ".sty"]}] {
        if {[file::searchFor $sty $pat 0] != ""} { return 1}
      }
    }
    global TeXmodeVars
    if {$TeXmodeVars(warnIfPackageNotIncluded)} {
      global searchNoisily
      if {$searchNoisily} {beep}
      switch -- [askyesno -c "The '$pkg' package seems not to be included in this LaTeX document.  Shall I insert a 'usepackage' line?"] {
        "yes" {
          set w [win::Current]
          placeBookmark
          TeX::InsertInPreamble "\\usepackage\{$pkg\}\r"
          bringToFront $w
          returnToBookmark
        }
        "cancel" {
          return 0
        }			
      }
    }
  }
  return 1
}

proc TeX::InsertInPreamble {text} {
  if {[selectPatternInFile [TeX_currentBaseFile] {\\documentclass(\[[^]]*\])?\{[^\}]*\}} ]} {
    goto [nextLineStart [selEnd]]
    insertText $text
    return 1
  } else {
    alertnote "Can't find the document preamble '\\documentclass...'"
    return 0
  }
}

proc TeX::Format {menu item} {TeX::SetFormat $item}
# Called when the user chooses a TeX format in the "TeX Format"
# menu. Either inserts the name of the format on the first line of the file
# or update mode prefs and mark the format in the menu.
proc TeX::SetFormat {text} {
  global TeXmodeVars
  
  if {!$TeXmodeVars(noFormatNameInFile)} {
    # Insert the name of the format on the first line
    file::openQuietly [TeX_currentBaseFile]
    set first [getText [minPos] [lineEnd [minPos]]]
    if {[regexp "^%&\[^:\t\]*$" $first]} {
      # only replace if different
      if {$first != "%&$text"} {replaceText [minPos] $p "%&$text"}
    } else {
      goto [minPos]
      insertText "%&$text\r"
    }
    select [minPos] [nextLineStart [minPos]]
  } else {
    # Just memorize the format name
    set TeXmodeVars(nameOfTeXFormat) "$text"
    prefs::modifiedModeVar nameOfTeXFormat TeX
    foreach itm $TeXmodeVars(availableTeXFormats) {
      markMenuItem -m {TeX Format} $itm off
    }
    markMenuItem -m {TeX Format} $text on
  }
}

## 
 # -------------------------------------------------------------------------
 #	 
 # "TeXOptionTitlebar"	--
 #	
 #  Invoked by option-clicking on the title-bar.  List either all files
 #  in the current fileset (if we're part of one), or just all
 #  input/include files.  Select one to open it in a window.
 # -------------------------------------------------------------------------
 ##
proc TeX::OptionTitlebar {} {
  set win [win::Current]
  if {[set fset [isWindowInFileset $win "tex"]] == ""} {
    # find includes directly
    set cid [scancontext create]
    set lines {}
    # copes with various commands with '[]' optional
    global TeX::subFileCommmands
    scanmatch $cid "^\[ \t\]*\\\\([join ${TeX::subFileCommmands} |])(\\\[\[^\]\]*\\\])?\{([^\r\n]*)\}"  {
      #eval lappend lines [split $matchInfo(submatch2) ,]
      lappend lines [list $matchInfo(submatch0) [split $matchInfo(submatch2) ,]]
    }
    set fid [alphaOpen $win "r"]
    scanfile $cid $fid
    close $fid
    scancontext delete $cid
    if {$lines != ""} {
      foreach l $lines {
        set type [lindex $l 0]
        foreach f [lindex $l 1] {
          switch -- $type {
            "include" {
              lappend files "${f}.tex"
            }
            "input" {
              if ![catch {findTeXFile $f ".tex"} f] {
                set f [file tail $f]
                lappend files $f
              }
            }
            "bibliography" {
              lappend files "${f}.bib"
            }
            "bibliographystyle" {
              lappend files "${f}.bst"
            }
            "usepackage" {
              lappend files "${f}.sty"
            }
            "RequirePackage" {
              lappend files "${f}.sty"
            }
            default {
              lappend files "${f}"
            }
          }
        }
      }
      return $files
    } else {
      # just use files in this directory instead, so signal error
      return [::OptionTitlebar]
    }
  } else {
    set lines {}
    global fileSetsExtra gfileSets
    lappend lines [file tail [lindex $gfileSets($fset) 0]]
    foreach f $fileSetsExtra($fset) { 
      if {[lindex $f 0] == "tex"} {
        lappend lines "[string range {      } 1 [expr {2*[lindex $f 1]}]][lindex $f 2]" 
      }
    }
    return $lines
  }
}
# used by the above proc
set TeX::subFileCommmands [list \
 include usepackage RequirePackage input bibliography bibliographystyle \
 ]


## 
 # -------------------------------------------------------------------------
 #	 
 # "TeXOptionTitlebarSelect" --
 #	
 #  Procedure called when you select an item from the option pop-up
 #  menu.  We just open the file if we can.  If there isn't a fileset,
 #  this procedure could fail.
 # -------------------------------------------------------------------------
 ##
proc TeX::OptionTitlebarSelect {fname} {
  set fname [string trim $fname]
  if {[set fset [isWindowInFileset "" "tex"]] == ""} {
    switch -- [file extension $fname] {
      ".bib" -
      ".bst" {
        Bib::openFile $fname
      }
      default {
        openTeXFile $fname
      }
    }
  } else {
    file::openQuietly [texfs_awkwardGetFile $fset $fname]
  }
}

proc TeX_currentBaseFile {{currentWin ""}} {
  if {$currentWin == ""} {
    global win::Active
    foreach f [set win::Active] {
      if {[file exists $f]} {
        set currentWin $f
        break
      }
    }
  }
  if {[set currentProj [isWindowInFileset $currentWin "tex"]] != ""} {
    return [texFilesetBaseName $currentProj]
  } else {
    return $currentWin
  }
}
## 
 # -------------------------------------------------------------------------
 #	 
 # "TeXEnsureSearchPathSet" --
 #	
 #  Make sure TeX mode has built our search path, so we can find
 #  bibliography files.  Perhaps we should have our own variable for
 #  these?
 # -------------------------------------------------------------------------
 ##
proc TeXEnsureSearchPathSet {} {
  global AllTeXSearchPaths
  if { [llength $AllTeXSearchPaths] == 0 } {
    message "building TeX search path"
    set AllTeXSearchPaths [buildTeXSearchPath]
    message ""
  }
}


## 
 # -------------------------------------------------------------------------
 # 
 # "findTeXFile" --
 # 
 #  Find a TeX file with given default extension if it exists anywhere on
 #  the TeX search path.
 # -------------------------------------------------------------------------
 ##
proc findTeXFile {file {ext ""}} {
  global AllTeXSearchPaths mode file::separator
  # Determine absolute file specification
  # Ignore $ext if $file already has an extension
  if {[string length [file extension $file]] == 0} {
    append file $ext
  }
  set filename [file::absolutePath $file]
  if {[file exists $filename]} { 
    return $filename
  }
  TeXEnsureSearchPathSet
  foreach folder $AllTeXSearchPaths {
    set filename [file join $folder $file]
    if {[file exists $filename]} {
      return $filename
    }
  }
  
  if { $mode == "TeX" } {
    # find '\graphicspath{{path}{path}{path}}' if it exists
    if {![catch {search -s -f 1 -r 1 {\\graphicspath\{[^\r\n]*\}} [minPos]} l]} {
      
      set l [getText [lindex $l 0] [lindex $l 1]]
      if {![regexp -indices {\{} $l ind]} {
        error "Malformed \\graphicspath"
      }
      set l [string range $l [expr [lindex $ind 0] + 1] end]
      set nesting 1
      set currentPath [file dirname [win::Current]]
      set graphicsPaths {}
      while {$nesting && [regexp -indices {[\{\}]} $l ind]} {
        switch -- [string index $l [set ind [lindex $ind 0]]] {
          "\{" {
            incr nesting
          }
          "\}" {
            if {$nesting == 2} {
              # graphicspath must have trailing file::separator, but we don't want it
              set graphicsPath [file dirname [string range $l 0 [expr $ind - 1]]]
              if {[string index $graphicsPath 0] == ${file::separator}} {
                # \graphicspath is relative to input file
                lappend graphicsPaths "$currentPath$graphicsPath"
              } else {
                # \graphicspath is absolute
                lappend graphicsPaths $graphicsPath
              }
            } 
            incr nesting -1
          }
        }
        set l [string range $l [incr ind] end]
      }  
      foreach folder $graphicsPaths {
        set filename [file join $folder $file]
        if {[file exists $filename]} {
          return $filename
        }
      }
    }
  }
  
  # Try recursing...
  foreach folder $AllTeXSearchPaths {
    foreach contents [file::recurse $folder] {
      if {[file tail $contents] == $file} {
        return $contents
      }
    }
  }
  
  error "File not found."
}
proc openTeXFile {file {ext ""}} {
  if {[catch {findTeXFile $file $ext} f]} {
    beep
    error "can't find TeX input file \"$file\""
  } else {
    file::openQuietly $f
    message $f
  }
}

# Return a list of folders in which to search for TeX input files, 
# including the TeXInputs folder (if it exists) and any folders of 
# the form "*input*" in the TeX application directory.  The current
# folder is not included in the list.  Default search depth is two
# levels deep.
#
# (Note: The TeXInputs folder is assigned from the AppPaths submenu.)
#
proc buildTeXSearchPath {{depth 2}} {
  global TeXmodeVars texSig
  
  set folders {}
  # The local 'TeXSearchPath' folder:
  if {[info exists TeXmodeVars(TeXSearchPath)] && \
   [string length $TeXmodeVars(TeXSearchPath)] > 0} { 
    eval lappend folders $TeXmodeVars(TeXSearchPath)
    # Search subfolders $depth levels deep:
    eval lappend folders [file::hierarchy $TeXmodeVars(TeXSearchPath) $depth]
  }
  
  # Any "*inputs*" folders in the TeX application folder:
  if {[info exists texSig] && [string length $texSig] > 0} { 
    set TeXDir [file dirname [nameFromAppl $texSig]]
    # Problem:  'glob' is case sensitive, macos isn't !
    foreach folder [glob -nocomplain -type d -dir $TeXDir "*\[Ii\]nputs*"] {
      lappend folders $folder
      # Search subfolders $depth levels deep:
      eval lappend folders [file::hierarchy $folder $depth]
    }
    # Now try any folders within a subfolder called 'inputs'
    foreach folder [glob -nocomplain -type d -dir $TeXDir -join * "*\[Ii\]nputs*"] {
      lappend folders $folder
      # Search subfolders $depth levels deep:
      eval lappend folders [file::hierarchy $folder $depth]
    }
  }
  
  return $folders
}


# Extend the argument around the position $from.
# (Args are delimited by commas or curly-braces.)
proc TeXExtendArg {from {to 0}} {
  if {$to == 0} { set to $from }
  set result [list $from $to]
  if {![catch {search -f 0 -r 1 -s -m 0 "\[,\{\]" $from} mtch0]} {
    if {![catch {search -f 1 -r 1 -s -m 0 "\[,\}\]" $to} mtch1]} {
      set from [lindex $mtch0 1]
      set to [lindex $mtch1 0]
      ## Embedded braces in the arg probably mean that the user
      ## clicked outside a valid command argument
      if {[regexp "\[\{\}\]" [getText $from $to]] == 0} {
        set result [list $from $to]
      }
    }
  }
  return $result
}

# Find a LaTeX command with arguments in either direction.
# (see findCommandWithArgs in latexMacros.tcl)
# This version returns the positions at which the command options 
# and arguments start, as well.
proc findCommandWithParts {pos direction} {
  set searchString {\\([^@a-zA-Z\t\n\r]|@?[a-zA-Z]+\*?)(\[[^]]*\])*({[^{}]*})?}
  if {![catch {search -s -f $direction -r 1 $searchString $pos} mtch]} {
    set beg [lindex $mtch 0]
    set end [lindex $mtch 1]
    regexp -indices -- $searchString [getText $beg $end] all cmd opt arg
    set opt [pos::math $beg + [lindex $opt 0]]
    set arg [pos::math $beg + [lindex $arg 0]]
    return [list $beg $opt $arg $end]
  } else {
    return ""
  }
}

#--------------------------------------------------------------------------
# Insertion routines:
#--------------------------------------------------------------------------

# Shift each line of $text to the right by inserting a string of
# $whitespace characters at the beginning of each line, returning
# the resulting string.
proc shiftTextRight {text whitespace} {
  return [doPrefixText "insert" $whitespace $text]
}

# Return an "indented carriage return" if any character preceding 
# the insertion point (on the same line) is a non-whitespace 
# character.  Otherwise, return the null string.
proc openingCarriageReturn {} {
  set pos [getPos]
  set end $pos
  set start [lineStart $pos]
  set text [getText $start $end]
  if {[is::Whitespace $text]} {
    return ""
  } else {
    return "\r"
  }
}
# Return an "indented carriage return" if any character following 
# the insertion point (on the same line) is a non-whitespace 
# character.  Otherwise, return the null string.
proc closingCarriageReturn {} {
  set pos [selEnd]
  if {[isSelection] && ([pos::compare $pos == [lineStart $pos]])} {
    return "\r"
  } else {
    set start $pos
    set end [nextLineStart $start]
    set text [getText $start $end]
    if {[is::Whitespace $text]} {
      return ""
    } else {
      return "\r"
    }
  }
}

# Insert an object at the insertion point. If there is a selection and the 
# global variable 'deleteObjNoisily' is false, quietly delete the selection 
# (like 'paste'). Otherwise, prompt the user for the appropriate action. 
# Returns true if the object is ultimately inserted, and false if the 
# user cancels the operation. 
proc insertObject {objectName} {
  global TeXmodeVars
  if {[isSelection]} {
    if {$TeXmodeVars(deleteObjNoisily)} {
      switch -- [askyesno -c "Delete selection?"] {
        "yes" {deleteText [getPos] [selEnd]}
        "no" {backwardChar}
        "cancel" {return 0}
      }
    } else {
      deleteText [getPos] [selEnd]
    }
  }
  elec::Insertion $objectName
  return 1
}

# Builds and returns a LaTeX environment, that is, a \begin...\end 
# pair, given the name of the environment, an argument string, 
# and the environment body.  The body should be passed to this 
# procedure fully formatted, including indentation.
proc buildEnvironment {envName envArg envBody trailingComment} {
  append begStruct "\\begin{" $envName "}" $envArg
  append endStruct "\\end{" $envName "}$trailingComment"
  return [buildStructure $begStruct $envBody $endStruct]
}
# Builds and returns a fully-formed structure, a string of the form
#
#   <begStruct>
#     <bodyStruct>
#   <endStruct>
#
# For example,
#
#   buildStructure "if {} {" "\t\r" "}"
#
# returns a Tcl if-template.
#
proc buildStructure {begStruct bodyStruct endStruct} {
  append structure [openingCarriageReturn] \
   $begStruct "\r" $bodyStruct \
   $endStruct [closingCarriageReturn]
  return $structure
}

# Inserts a LaTeX environment with the specified name, argument, 
# and body at the insertion point.  Deletes the current selection 
# quietly if the global variable 'deleteEnvNoisily' is false; 
# otherwise the user is prompted for directions.  Returns true if the 
# environment is ultimately inserted, and false if the user cancels 
# the operation.
proc insertEnvironment {envName envArg envBody} {
  global TeXmodeVars
  if { [isSelection] } {
    if { $TeXmodeVars(deleteEnvNoisily) } {
      switch -- [askyesno -c "Delete selection?"] {
        "yes" {}
        "no" {backwardChar}
        "cancel" {return 0}
      }
    }
  }
  append begStruct "\\begin{" $envName "}" $envArg
  append endStruct "\\end{" $envName "}"
  insertStructure $begStruct $envBody $endStruct
  return 1
}
# Inserts a structure at the insertion point.  Positions the cursor 
# at the beginning of the structure, leaving any subsequent action 
# to the calling procedure.  Deletes the current selection quietly.
proc insertStructure {begStruct bodyStruct endStruct} {
  set start [getPos]
  set end [selEnd]
  #set body [shiftTextRight $bodyStruct [text::indentString $start]]
  elec::ReplaceText $start $end [buildStructure $begStruct $bodyStruct $endStruct]
}


# Inserts an environment with the given name, argument, and body at 
# the insertion point.  If there is currently a selection, cut and 
# paste it into the body of the new environment, maintaining proper 
# indentation; otherwise, insert a tab stop into the body of the
# environment.  Returns true if there is a selection, and false 
# otherwise.
proc wrapEnvironment {envName envArg envBody} {
  append begStruct "\\begin{" $envName "}" $envArg
  append endStruct "\\end{" $envName "}"
  return [wrapStructure $begStruct $envBody $endStruct]
}
# Inserts a structure at the insertion point.  Positions the cursor 
# at the beginning of the structure, leaving any subsequent action 
# to the calling procedure.  If there is currently a selection, cut 
# and paste it into the body of the new environment, maintaining proper 
# indentation; otherwise, insert a tab stop into the body of the
# environment.  Returns true if there is a selection, and false 
# otherwise.
proc wrapStructure {begStruct bodyStruct endStruct} {
  set start [getPos]
  set end [selEnd]
  if {[isSelection]} {
    set text [getSelect]
    set textLen [string length $text]
    if { [string index $text [expr {$textLen-1}]] != "\r" } {
      append text "\r"
    }
    set body [shiftTextRight $text \t]
    append body $bodyStruct
    set returnFlag 1
  } else {
    append body "\t\r" $bodyStruct
    set returnFlag 0
  }
  elec::ReplaceText $start $end [buildStructure $begStruct $body $endStruct]
  return $returnFlag
}

# A generic call to 'wrapEnvironment' used throughout latex.tcl:
proc doWrapEnvironment {envName} {
  if {[wrapEnvironment $envName "" ""]} {
    set msgText "selection wrapped"
  } else {
    set msgText "enter body of $envName environment"
  }
  message $msgText
}
# A generic call to 'wrapStructure':
proc doWrapStructure {begStruct bodyStruct endStruct} {
  if {[wrapStructure $begStruct $bodyStruct $endStruct]} {
    set msgText "selection wrapped"
  } else {
    set msgText "enter body of structure"
  }
  message $msgText
}

# Inserts a structured document template at the insertion point.  
# Three arguments are required:  the class name of the document, a 
# preamble string, and a string containing the body of the document.  
# If the preamble is null, a generic \usepackage statement is 
# inserted; otherwise, the preamble is inserted as is.  This routine 
# does absolutely no error-checking (this is totally left up to the 
# calling procedure) and returns nothing.
proc insertDocument {className preamble docBody} {
  set docStr "\\documentclass\[\]{$className}\r"
  if {$preamble == ""} {
    append docStr "\\usepackage\[\]{}\r\r\r\r"
  } else {
    append docStr $preamble
  }
  append docStr [buildEnvironment "document" "" $docBody "\r"]
  set start [getPos]
  set end [selEnd]
  elec::ReplaceText $start $end $docStr
  return
}

# Inserts a document template at the insertion point given the 
# class name of the document to be inserted.  If ALL of the current
# document is selected, then the routine wraps the text inside a
# generic document template.  If the file is empty, a bullet is 
# inserted in place of the document body.  If neither of these 
# conditions is true, no action is taken.  Returns true if 
# wrapping occurs, and false otherwise.
proc wrapDocument {className} {
  global TeXmodeVars
  if { [isEmptyFile] } {
    append body "\r\r\r"
    # 		set returnFlag 0
  } else {
    if {[isSelectionAll]} {
      set text [getSelect]
      append body "\r$text\r"
      # 			set returnFlag 1
    } else {
      if { $TeXmodeVars(searchNoisily) } {beep}
      alertnote "nonempty file:  delete text or \'Select All\'\
       from the Edit menu"
      return 0
    }
  }
  set docStr "\\documentclass\[\]{$className}\r"
  append docStr "\\usepackage\[\]{}\r\r\r\r"
  append docStr [buildEnvironment "document" "" $body "\r"]
  set start [getPos]
  set end [selEnd]
  elec::ReplaceText $start $end $docStr
  # 	return $returnFlag
  return 1
}

#--------------------------------------------------------------------------
# Booleans to determine the location of the insertion point
#--------------------------------------------------------------------------

# Return true if the insertion point is before the preamble, and 
# false otherwise.   Define "before the preamble" to be all text to 
# the left of "\" in "\documentclass".
proc isBeforePreamble {} {
  set searchString "\\documentclass"
  set searchResult [search -s -f 1 -r 0 -n $searchString [getPos]]
  if {[llength $searchResult]} {
    return 1
  } else {
    return 0
  }
}

# Return true if the insertion point is in the preamble, and false
# otherwise.  Define "preamble" to be all text to the left of "\" 
# in "\begin{document}", but not before the "\" in "\documentclass".
proc isInPreamble {} {
  set searchString "\\begin\{document\}"
  set searchResult [search -s -f 1 -r 0 -n $searchString [getPos]]
  if {[llength $searchResult]} {
    return 1
  } else {
    return 0
  }
}

# Return true if the insertion point is in the document environment,
# and false otherwise.  Define "document" to be all text between 
# "\begin{document}" and "\end{document}", exclusive.
proc isInDocument {} {
  set pos [getPos]
  set searchString "\\begin\{document\}"
  # adjust for the length of the search string:
  set len [string length $searchString]
  if {[pos::compare $pos < [pos::math [minPos] + $len]]} {
    return 0
  }
  set searchResult [search -s -f 0 -r 0 -n $searchString [pos::math $pos - $len]]
  if {[llength $searchResult]} {
    set searchString "\\end\{document\}"
    set searchResult [search -s -f 1 -r 0 -n $searchString $pos]
    if {[llength $searchResult]} {
      return 1
    } else {
      return 0
    }
  } else {
    return 0
  }
}

# Return true if the insertion point is after the document
# environment, and false otherwise.  Define "after the document 
# environment" to be all text to the right of "}" in "\end{document}".
proc isAfterDocument {} {
  set pos [getPos]
  set searchString "\\end\{document\}"
  set searchResult [search -s -f 0 -r 0 -n $searchString $pos]
  if {[llength $searchResult]} {
    set pos1 [lindex $searchResult 0]
    set pos2 [lindex $searchResult 1]
    if {[pos::compare $pos1 <= $pos] && [pos::compare $pos < $pos2]} {
      return 0
    } else {
      return 1
    }
  } else {
    return 0
  }
}

proc isInMathMode {} {
  set pos [getPos]
  # Check to see if in LaTeX math mode:
  set searchString {\\[()]}
  set searchResult1 [search -s -f 0 -r 1 -n $searchString [pos::math $pos - 1]]
  if {[llength $searchResult1]} {
    set delim1 [eval getText $searchResult1]
  } else {
    set delim1 "none"
  }
  set searchResult2 [search -s -f 1 -r 1 -n $searchString $pos]
  if {[llength $searchResult2]} {
    set delim2 [eval getText $searchResult2]
  } else {
    set delim2 "none"
  }
  set flag1 [expr [string match {none} $delim1] && [string match {\\)} $delim2]]
  set flag2 [expr [string match {\\(} $delim1] && [string match {none} $delim2]]
  set flag3 [expr [string match {\\(} $delim1] && [string match {\\)} $delim2]]
  set flag4 [expr [string match {\\(} $delim1] && [string match {\\(} $delim2]]
  set flag5 [expr [string match {\\)} $delim1] && [string match {\\)} $delim2]]
  if {$flag3} {
    return 1
  } elseif {$flag1 || $flag2 || $flag4 || $flag5} {
    set messageString "unbalanced math mode delimiters"
    beep
    alertnote $messageString
    error "isInMathMode:  $messageString"
  }
  # Check to see if in LaTeX displaymath mode:
  set searchString {[^\\]\\\[|\\\]}
  set searchResult1 [search -s -f 0 -r 1 -n $searchString [pos::math $pos - 1]]
  if {[llength $searchResult1]} {
    set begPos [lindex $searchResult1 0]
    set endPos [lindex $searchResult1 1]
    if {[lookAt [pos::math $endPos - 1]] == "\["} {
      set delim1 [getText [pos::math $begPos + 1] $endPos]
    } else {
      set delim1 [eval getText $searchResult1]
    }
  } else {
    set delim1 "none"
  }
  set searchResult2 [search -s -f 1 -r 1 -n $searchString $pos]
  if {[llength $searchResult2]} {
    set begPos [lindex $searchResult2 0]
    set endPos [lindex $searchResult2 1]
    if {[lookAt [pos::math $endPos - 1]] == "\["} {
      set delim2 [getText [pos::math $begPos + 1] $endPos]
    } else {
      set delim2 [eval getText $searchResult2]
    }
  } else {
    set delim2 "none"
  }
  set flag1 [expr [string match {none} $delim1] && [string match {\\\]} $delim2]]
  set flag2 [expr [string match {\\\[} $delim1] && [string match {none} $delim2]]
  set flag3 [expr [string match {\\\[} $delim1] && [string match {\\\]} $delim2]]
  set flag4 [expr [string match {\\\[} $delim1] && [string match {\\\[} $delim2]]
  set flag5 [expr [string match {\\\]} $delim1] && [string match {\\\]} $delim2]]
  if {$flag3} {
    return 1
  } elseif {$flag1 || $flag2 || $flag4 || $flag5} {
    set messageString "unbalanced math mode delimiters"
    beep
    alertnote $messageString
    error "isInMathMode:  $messageString"
  }
  # Check to see if in math environment:
  set envName [extractCommandArg [eval getText [searchEnvironment]]]
  global mathEnvironments
  if {[lsearch -exact $mathEnvironments $envName] == -1} {return 0} {return 1}
}

#--------------------------------------------------------------------------
# Routines to dissect LaTeX commands
#--------------------------------------------------------------------------

# Given a LaTeX command string, extract and return the command name.
proc extractCommandName {commandStr} {
  if {[regexp {\\([^@a-zA-Z\t\n\r]|@?[a-zA-Z]+\*?)} $commandStr dummy commandName]} {
    return $commandName
  } else {
    return ""
  }
}

# Given a LaTeX command string with at most one required argument and
# no embedded carriage returns, extract and return the argument.
# (Note:  this proc needs more testing.)
proc extractCommandArg {commandStr} {
  if {[regexp {\{(.*)\}} $commandStr dummy arg]} {
    return $arg
  } else {
    return ""
  }
}

#--------------------------------------------------------------------------
# An environment search routine
#--------------------------------------------------------------------------

# Search for the closest surrounding environment and return a list 
# of two positions if the environment is found; otherwise return the 
# empty list.  Assumes the LaTeX document is syntactically correct.   
# The command 'eval select [searchEnvironment]' selects the "\begin" 
# statement (with argument) of the surrounding environment.
proc searchEnvironment {} {
  watchCursor
  set pos [getPos]
  if {[pos::compare $pos == [minPos]]} {
    return [list]
  }
  # adjust position if insertion point is contained in "\end{...}"
  set searchPos [pos::math $pos - 1]
  set searchString {\\end\{[^\{\}]*\}}
  set searchResult [search -s -f 0 -r 1 -n $searchString $searchPos]
  if {[llength $searchResult]} {
    set pos1 [lindex $searchResult 0]
    set pos2 [lindex $searchResult 1]
    if {[pos::compare $pos1 < $pos] && [pos::compare $pos < $pos2]} {
      set searchPos [pos::math $pos1 - 1]
    }
  }
  # begin reverse search:
  set searchString {\\(begin|end)\{[^\{\}]*\}}
  set searchResult [search -s -f 0 -r 1 -n $searchString $searchPos]
  if {[llength $searchResult]} {
    set text [eval getText $searchResult]
  } else {
    return [list]
  }
  set depthCounter 0
  set commandName [extractCommandName $text]
  while {$commandName == "end" || $depthCounter > 0} {
    if {$commandName == "end"} {
      incr depthCounter
    } else {
      incr depthCounter -1
    }
    set searchPos [pos::math [lindex $searchResult 0] - 1]
    set searchResult [search -s -f 0 -r 1 -n $searchString $searchPos]
    if {[llength $searchResult]} {
      set text [eval getText $searchResult]
    } else {
      return [list]
    }
    set commandName [extractCommandName $text]
  }
  return $searchResult
}

#--------------------------------------------------------------------------
# Misc:
#--------------------------------------------------------------------------

# Returns true if the entire window is selected, and false otherwise.
proc isSelectionAll {} {
  return [expr {([pos::compare [getPos] == [minPos]]) && ([pos::compare [selEnd] == [maxPos]])}]
}

# Checks to see if the current window is empty, except for whitespace.
proc isEmptyFile {} {
  return [is::Whitespace [getText [minPos] [maxPos]]]
}

# If there is a selection, make sure it's uppercase.  Otherwise, 
# check to see if the character after the insertion point is uppercase.
proc isUppercase {} {
  if {[isSelection]} {
    set text [getSelect]
  } else {
    set text [lookAt [getPos]]
  }
  return [expr {[string toupper $text] == $text}]
}

# If there is a selection, make sure it's alphabetic.  Otherwise, 
# check to see if the character after the insertion point is alphabetic.
proc isAlphabetic {} {
  if {[isSelection]} {
    set text [getSelect]
  } else {
    set text [lookAt [getPos]]
  }
  return [regexp {^[a-zA-Z]+$} $text]
}

proc checkMathMode {procName mathModeFlag} {
}

########### New stuff FBO 2001-09 ############
# This proc is called when the user chooses a TeX
# program in the "TeX program" menu
proc TeX::ChooseProgram {menu item} {
  TeX::SetProgram $item
}

# New TeX program choosen, update mode prefs and
# mark corresponding program in the menu.
proc TeX::SetProgram {text} {
  global TeXmodeVars
  
  set TeXmodeVars(nameOfTeXProgram) $text
  prefs::modifiedModeVar nameOfTeXProgram TeX
  foreach itm $TeXmodeVars(availableTeXPrograms) {
    markMenuItem -m {TeX program} $itm off
  }
  markMenuItem -m {TeX program} $text on
}

proc TeX::ChooseStyle {menu item} {
  global TeX::indexStyle TeX::glossaryStyle
  
  switch -- $item {
    "Index style " {set what "index"}
    "Glossary style " {set what "glossary"}
    "No index style" {
      set TeX::indexStyle ""
      buildLaTeXMenuQuietly
      return
    }
    "No glossary style" {
      set TeX::glossaryStyle ""
      buildLaTeXMenuQuietly
      return
    }
    default {
      alertnote "Unknown menu item in MakeIndex styles menu"
      return
    }
  }
  
  if {[catch {file tail [getfile "Choose $what style"]} "TeX::${what}Style"] == 0} {
    buildLaTeXMenuQuietly
  }
}
